import grails.orm.PagedResultList
import org.compass.core.engine.SearchEngineQueryParseException

/**
* Service class that encapsulates the business logic for InventoryItem searches.
*/
class InventoryItemSearchService {

    boolean transactional = false

    def dateUtilService
    def messageSource

    def paramsMax = 100000

    /**
    * Selects and returns the correct search results based on the supplied quickSearch.
    * @param params The request params, may contain params.quickSearch string to specify the search.
    * @param locale The locale to use when generating result.message.
    */
    def getQuickSearch(params, locale) {
        def result = [:]
        result.quickSearch = params.quickSearch ?: "all"

        def getMessage = { Map m ->
            messageSource.getMessage(m.code, m.args == null ? null : m.args.toArray(), locale)
        }

        switch (result.quickSearch) {
            case "inventoryBelowReorder":
                result.inventoryItemList = getInventoryBelowReorder(params)
                if(result.inventoryItemList.totalCount > 0)
                    result.message = getMessage(code:"inventoryItem.search.text.below.reorder.description")
                else
                    result.message = getMessage(code:"inventoryItem.search.text.below.reorder.none.found")
                break
            case "inventoryBelowReorderAll":
                result.inventoryItemList = getInventoryBelowReorder(params, false)
                if(result.inventoryItemList.totalCount > 0)
                    result.message = getMessage(code:"inventoryItem.search.text.below.reorder.all.description")
                else
                    result.message = getMessage(code:"inventoryItem.search.text.below.reorder.none.found")
                break
            case "recentlyUsed":
                result.daysBack = params.daysBack?.isInteger() ? params.daysBack.toInteger() : 14
                result.inventoryItemList = getRecentlyUsed(params, result.daysBack)
                if(result.inventoryItemList.totalCount > 0)
                    result.message = getMessage(code:"inventoryItem.search.text.recently.used.description", args:[result.daysBack])
                else
                    result.message = getMessage(code:"inventoryItem.search.text.recently.used.none.found", args:[result.daysBack])
                break
            default:
                result.inventoryItemList = getAll(params)
                if(result.inventoryItemList.totalCount > 0)
                    result.message = getMessage(code:"inventoryItem.search.text.all.description")
                else
                    result.message = getMessage(code:"inventoryItem.search.text.all.none.found")
                break
        } // switch.

        // Success.
        return result

    } // getQuickSearch

    /**
    * Get all inventory items.
    * @param params The request params.
    */
    def getAll(params) {
        params.max = Math.min(params?.max?.toInteger() ?: 10, paramsMax)
        params.offset = params?.offset?.toInteger() ?: 0
        params.sort = params?.sort ?: "name"
        params.order = params?.order ?: "asc"

        def inventoryItemList = InventoryItem.createCriteria().list(
            max: params.max,
            offset: params.offset,
            sort: params.sort,
            order: params.order) {
            } // createCriteria
    } // getAll

    /**
    * List inventory items that are below reorder point.
    * @param params The request params.
    * @param onlyReorderEnabled Only include items with reorder enabled, defaults to true.
    */
    def getInventoryBelowReorder(params, onlyReorderEnabled=true) {
        params.max = Math.min(params?.max?.toInteger() ?: 10, paramsMax)
        params.offset = params?.offset?.toInteger() ?: 0
        params.sort = params?.sort ?: "name"
        params.order = params?.order ?: "asc"

        def inventoryItemList = InventoryItem.createCriteria().list(
            max: params.max,
            offset: params.offset,
            sort: params.sort,
            order: params.order) {
                eq("isActive", true)
                if(onlyReorderEnabled)
                    eq("enableReorder", true)
                leProperty("unitsInStock", "reorderPoint")
            } // createCriteria
    } // getInventoryBelowReorder

    /**
    * Get a list of recently used inventory items.
    * @param params The request params.
    * @param daysBack The number of days back to get results for.
    */
    def getRecentlyUsed(params, daysBack) {
        def paginateParams = [:]
        paginateParams.max = Math.min(params?.max?.toInteger() ?: 10, paramsMax)
        paginateParams.offset = params?.offset?.toInteger() ?: 0

        def sort = "inventoryItem." + (params?.sort ?: "name")
        def order = params?.order == "desc" ? "desc" : "asc"
        def orderBy = " order by " + sort + ' ' + order

        def namedParams = [:]
        namedParams.startOfDay = dateUtilService.today - daysBack

        def baseQuery = "from InventoryItem as inventoryItem \
                                        left join inventoryItem.inventoryMovements as inventoryMovement \
                                        where (inventoryItem.isActive = true \
                                                    and inventoryMovement.date > :startOfDay \
                                                    and inventoryMovement.inventoryMovementType = 1 \
                                                    )"

        def searchQuery = "select distinct inventoryItem " + baseQuery + orderBy
        def list = InventoryItem.executeQuery(searchQuery, namedParams, paginateParams)

        def countQuery = "select count(distinct inventoryItem) as inventoryItemCount " + baseQuery
        def totalCount = InventoryItem.executeQuery(countQuery, namedParams)[0].toInteger()

        def inventoryItemInstanceList = new PagedResultList(list, totalCount)
        return inventoryItemInstanceList
    } // getRecentlyUsed

    /**
    * Get a list of inventory items by search text.
    * @param params The request params.
    * @param locale The locale to use when generating result.message.
    */
    def getTextSearch(params, locale) {
        def result = [:]
        result.searchText = params.searchText.trim() ?: ""

        def getMessage = { Map m ->
            messageSource.getMessage(m.code, m.args == null ? null : m.args.toArray(), locale)
        }

        params.max = Math.min(params?.max?.toInteger() ?: 10, paramsMax)
        params.offset = params?.offset?.toInteger() ?: 0
        params.sort = params?.sort ?: "id"
        params.order = params?.order ?: "asc"

        // Build searchableParams.
        // Do not include params.sort, since not all properites are indexed.
        def searchableParams = [:]
        searchableParams.max = params.max
        searchableParams.offset = params.offset
        searchableParams.reload = true
        searchableParams.defaultOperator =  'or'
        def properitesList = []
        if(params.searchName)
            properitesList << '$/InventoryItem/name'
        if(params.searchDescription)
            properitesList << '$/InventoryItem/description'
        if(params.searchComment)
            properitesList << '$/InventoryItem/comment'
        if(params.searchLocation)
            properitesList << '$/InventoryItem/inventoryLocation/name'
        if(params.searchGroup)
            properitesList << '$/InventoryItem/inventoryGroup/name'
        if(params.searchSpareFor) {
            properitesList << '$/InventoryItem/spareFor/name'
            properitesList << '$/InventoryItem/spareFor/description'
            properitesList << '$/InventoryItem/spareFor/comment'
        }
        if(properitesList)
            searchableParams.properties = properitesList

        // Require isActive and isObsolete.
        def queryString = '( '+result.searchText+' )'+ ' AND isActive:"true" AND isObsolete:"false"'

        // Perform the searchable query.
        try {
            result.inventoryItemList = InventoryItem.search(queryString, searchableParams)

            // Would be nice if this worked.
//             result.inventoryItemList = InventoryItem.search(result.searchText, searchableParams) {
//                 must(term("isActive", true))
//                 must(term("isObsolete", false))
//             }

        } catch (e) {
            log.error e
            result.inventoryItemList = [:]
            result.inventoryItemList.results = []
            result.inventoryItemList.total = 0
        }

        // Sort the returned instances.
        if(params.sort != 'id') {
            if(params.order == 'asc') {
                if(params.sort == 'name' || params.sort == 'description')
                    result.inventoryItemList.results.sort { p1, p2 -> p1[params.sort].compareToIgnoreCase(p2[params.sort]) }
                else if(params.sort == 'inventoryGroup') {
                    result.inventoryItemList.results.sort { p1, p2 ->
                        p1.inventoryGroup.name.compareToIgnoreCase(p2.inventoryGroup.name)
                    }
                }
                else if(params.sort == 'unitsInStock')
                    result.inventoryItemList.results.sort {p1, p2 -> p1[params.sort]  <=> p2[params.sort] }
            } // asc.
            else {
                if(params.sort == 'name' || params.sort == 'description')
                    result.inventoryItemList.results.sort { p1, p2 -> p2[params.sort].compareToIgnoreCase(p1[params.sort]) }
                else if(params.sort == 'inventoryGroup') {
                    result.inventoryItemList.results.sort { p1, p2 ->
                        p2.inventoryGroup.name.compareToIgnoreCase(p1.inventoryGroup.name)
                    }
                }
                else if(params.sort == 'unitsInStock')
                    result.inventoryItemList.results.sort {p1, p2 -> p2[params.sort] <=> p1[params.sort]}
            } // desc.
        } // sort.

        // Create a PagedResultList.
        result.inventoryItemList = new PagedResultList(result.inventoryItemList.results, result.inventoryItemList.total)

        // Get the result message.
        if(result.inventoryItemList.totalCount > 0)
            result.message = getMessage(code:"inventoryItem.search.text.found", args: [result.searchText])
        else
            result.message = getMessage(code:"inventoryItem.search.text.none.found", args: [result.searchText])

        // Success.
        return result

    } // getTextSearch()

} // end class