import org.codehaus.groovy.grails.plugins.springsecurity.Secured
import org.codehaus.groovy.grails.commons.ConfigurationHolder
import com.zeddware.grails.plugins.filterpane.FilterUtils
import org.springframework.web.servlet.support.RequestContextUtils as RCU

@Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager'])
class InventoryItemDetailedController extends BaseController {

    def filterService
    def exportService
    def inventoryCsvService
    def inventoryItemService
    def inventoryItemSearchService
    def inventoryMovementService

    // the delete, save and update actions only accept POST requests
    static allowedMethods = [delete:'POST', save:'POST', update:'POST', useInventoryItem:'POST']

    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def index = { redirect(action:search, params:params) }

    /**
    * Set session.inventoryItemSearchParamsMax
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def setSearchParamsMax = {
        def max = 1000
        if(params.newMax.isInteger()) {
            def i = params.newMax.toInteger()
            if(i > 0 && i <= max)
                session.inventoryItemSearchParamsMax = params.newMax
            if(i > max)
                session.inventoryItemSearchParamsMax = max
        }
        forward(action: 'search', params: params)
    }

    /**
    * Display the import view.
    */
    def importInventory = {
    }

    /**
    * Handle the import save.
    */
    def importInventorySave = {
        def result = inventoryCsvService.importInventory(request)

        if(!result.error) {
            flash.message = g.message(code: "inventory.import.success")
            redirect(action:search)
            return
        }

        flash.errorMessage = g.message(code: result.error.code, args: result.error.args)
        redirect(action: importInventory)
    }

    /**
    * Export a csv template.
    * NOTE: IE has a 'validating' bug in dev mode that causes the export to take a long time!
    * This does not appear to be a problem once deployed to Tomcat.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def exportInventoryTemplate = {
        response.contentType = ConfigurationHolder.config.grails.mime.types["csv"]
        response.setHeader("Content-disposition", "attachment; filename=InventoryTemplate.csv")
        def s = inventoryCsvService.buildInventoryTemplate()
        render s
    }

    /**
    * Export a csv test file.
    */
    def exportInventoryExample = {
        response.contentType = ConfigurationHolder.config.grails.mime.types["csv"]
        response.setHeader("Content-disposition", "attachment; filename=InventoryExample.csv")
        def s = inventoryCsvService.buildInventoryExample()
        render s
    }

    /**
    * Export the entire inventory as a csv file.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def exportInventory = {

        def inventoryItemList = InventoryItem.list()

        response.contentType = ConfigurationHolder.config.grails.mime.types["csv"]
        response.setHeader("Content-disposition", "attachment; filename=Inventory.csv")
        def s = inventoryCsvService.buildInventory(inventoryItemList)
        render s
    }

    /**
    * Display the import view for purchases.
    */
    def importInventoryItemPurchases = {
    }

    /**
    * Handle the inventory purchases import save.
    */
    def importInventoryItemPurchasesSave = {
        def result = inventoryCsvService.importInventoryItemPurchases(request)

        if(!result.error) {
            flash.message = g.message(code: "inventory.import.success")
            redirect(action:search)
            return
        }

        flash.errorMessage = g.message(code: result.error.code, args: result.error.args)
        redirect(action: importInventoryItemPurchases)
    }

    /**
    * Search for Inventory items.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def search = {

        if(session.inventoryItemSearchParamsMax)
            params.max = session.inventoryItemSearchParamsMax

        // Protect filterPane.
        params.max = Math.min( params.max ? params.max.toInteger() : 10,  1000)

        def inventoryItemInstanceList = []
        def inventoryItemInstanceTotal
        def filterParams = com.zeddware.grails.plugins.filterpane.FilterUtils.extractFilterParams(params)
        def isFilterApplied = FilterUtils.isFilterApplied(params)

        // Restore search unless a new search is being requested.
        if(!params.quickSearch && !filterParams) {
            if(session.inventoryItemQuickSearch) {
                params.quickSearch = session.inventoryItemQuickSearch
                if(session.inventoryItemQuickSearchDaysBack)
                    params.daysBack = session.inventoryItemQuickSearchDaysBack.toString()
            }
            else if(session.inventoryItemSearchFilterParams) {
                session.inventoryItemSearchFilterParams.each() { params[it.key] = it.value }
                params.filter = session.inventoryItemSearchFilter
                isFilterApplied = FilterUtils.isFilterApplied(params)
            }
        }

        // Remember sort if supplied, otherwise try to restore.
        if(params.sort && params.order) {
             session.inventoryItemSearchSort = params.sort
             session.inventoryItemSearchOrder = params.order
        }
        else if(session.inventoryItemSearchSort && session.inventoryItemSearchOrder) {
            params.sort = session.inventoryItemSearchSort
            params.order = session.inventoryItemSearchOrder
        }

        if(isFilterApplied) {
            // filterPane:
            inventoryItemInstanceList = filterService.filter( params, InventoryItem )
            inventoryItemInstanceTotal = filterService.count( params, InventoryItem )
            filterParams = com.zeddware.grails.plugins.filterpane.FilterUtils.extractFilterParams(params)
            // Remember search.
            session.inventoryItemSearchFilterParams = new LinkedHashMap(filterParams)
            session.inventoryItemSearchFilter = new LinkedHashMap(params.filter)
            session.removeAttribute("inventoryItemQuickSearch")
            session.removeAttribute("inventoryItemQuickSearchDaysBack")
        }
        else {
            // Quick Search:
            if(!params.quickSearch) params.quickSearch = "all"
            def result = inventoryItemSearchService.getQuickSearch(params, RCU.getLocale(request))
            inventoryItemInstanceList = result.inventoryItemList
            inventoryItemInstanceTotal = result.inventoryItemList.totalCount
            params.message = result.message
            filterParams.quickSearch = result.quickSearch
            // Remember search.
            session.removeAttribute("inventoryItemSearchFilterParams")
            session.removeAttribute("inventoryItemSearchFilter")
            session.inventoryItemQuickSearch = result.quickSearch
            if(result.daysBack)
                session.inventoryItemQuickSearchDaysBack = result.daysBack
        }

        // export plugin:
        if(params?.format && params.format != "html") {

            def dateFmt = { date ->
                formatDate(format: "EEE, dd-MMM-yyyy", date: date)
            }

            String title
            if(params.quickSearch)
                title = params.message
            else
                title = "Filtered Inventory List."

            response.contentType = ConfigurationHolder.config.grails.mime.types[params.format]
            response.setHeader("Content-disposition", "attachment; filename=Inventory.${params.extension}")
            List fields = ["name",
                                "description",
                                "inventoryGroup",
                                "unitsInStock",
                                "reorderPoint",
                                "unitOfMeasure",
                                "inventoryLocation",
                                "inventoryLocation.inventoryStore"]
            Map labels = ["name": "Name",
                                "description": "Description",
                                "inventoryGroup": "Group",
                                "unitsInStock":"In Stock",
                                "reorderPoint":"Reorder Point",
                                "unitOfMeasure": "UOM",
                                "inventoryLocation": "Location",
                                "inventoryLocation.inventoryStore": "Store"]

            Map formatters = [:]
            Map parameters = [title: title, separator: ","]

            exportService.export(params.format,
                                                response.outputStream,
                                                inventoryItemInstanceList.sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) },
                                                fields,
                                                labels,
                                                formatters,
                                                parameters)
        }

        // Add some basic params to filterParams.
        filterParams.max = params.max
        filterParams.offset = params.offset?.toInteger() ?: 0
        filterParams.sort = params.sort ?: "name"
        filterParams.order = params.order ?: "asc"

        // Get some associatedProperty values for filterpane.
        def associatedPropertyValues = [:]
        def associatedPropertyMax = 10000
        associatedPropertyValues.inventoryLocationList = InventoryLocation.findAllByIsActive(true, [max:associatedPropertyMax, sort:'name'])
        associatedPropertyValues.assetList = Asset.findAllByIsActive(true, [max:associatedPropertyMax, sort:'name'])
        associatedPropertyValues.manufacturerList = Manufacturer.findAllByIsActive(true, [max:associatedPropertyMax, sort:'name'])
        associatedPropertyValues.supplierList = Supplier.findAllByIsActive(true, [max:associatedPropertyMax, sort:'name'])

        return[ inventoryItemInstanceList: inventoryItemInstanceList,
                        inventoryItemInstanceTotal: inventoryItemInstanceTotal,
                        filterParams: filterParams,
                        params: params,
                        associatedPropertyValues: associatedPropertyValues ]
    } // end search()

    /**
    * Simply assigns a passed in task id to a session variable and redirects to search.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def findInventoryItemForMovement = {
        if(!params.task?.id) {
            flash.message = "No task id supplied, please select a task then the inventory tab."
            redirect(controller: "taskDetailed", action: "search")
            return
        }

        session.inventoryMovementTaskId = params.task.id
        flash.message = "Please find and then select the inventory item."
        redirect(action: search)
    }

    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def show = {

        // In the case of an actionSubmit button, rewrite action name from 'index'.
        if(params._action_Show)
            params.action='show'

        def result = inventoryItemService.show(params)

        if(!result.error) {

            def model = [ inventoryItemInstance: result.inventoryItemInstance,
                                    inventoryMovementList: result.inventoryMovementList,
                                    inventoryMovementListTotal: result.inventoryMovementListTotal,
                                    inventoryMovementListMax: result.inventoryMovementListMax,
                                    inventoryItemPurchases: result.inventoryItemPurchases,
                                    inventoryItemPurchasesTotal: result.inventoryItemPurchasesTotal,
                                    showTab: result.showTab]

            if(session.inventoryMovementTaskId) {
                model.inventoryMovementInstance = new InventoryMovement()
                model.inventoryMovementInstance.task = Task.get(session.inventoryMovementTaskId)
                model.inventoryMovementInstance.quantity = 1
            }

            // Success.
            return model
        }

        flash.errorMessage = g.message(code: result.error.code, args: result.error.args)
        redirect(action:search)
    }

    def delete = {
        def result = inventoryItemService.delete(params)

        if(!result.error) {
            flash.message = g.message(code: "default.delete.success", args: ["InventoryItem", params.id])
            redirect(action:search)
            return
        }

        flash.errorMessage = g.message(code: result.error.code, args: result.error.args)

        if(result.error.code == "default.not.found") {
            redirect(action:search)
            return
        }

        redirect(action:show, id: params.id)
    }

    def edit = {

        // In the case of an actionSubmit button, rewrite action name from 'index'.
        if(params._action_Edit)
            params.action='edit'

        def result = inventoryItemService.edit(params)

        if(!result.error) {
            def possibleAlternateItems = inventoryItemService.getPossibleAlternateItems(result.inventoryItemInstance)
            def suppliers = Supplier.findAllByIsActive(true).sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) }
            def manufacturers = Manufacturer.findAllByIsActive(true).sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) }

            return [ inventoryItemInstance : result.inventoryItemInstance,
                            possibleAlternateItems: possibleAlternateItems,
                            suppliers: suppliers,
                            manufacturers: manufacturers]
        }

        flash.errorMessage = g.message(code: result.error.code, args: result.error.args)
        redirect(action:search)
    }

    def update = {
        def result = inventoryItemService.update(params)

        if(!result.error) {
            flash.message = g.message(code: "default.update.success", args: ["InventoryItem", params.id])
            redirect(action:show, id: params.id)
            return
        }

        if(result.error.code == "default.not.found") {
            flash.errorMessage = g.message(code: result.error.code, args: result.error.args)
            redirect(action:search)
            return
        }

        def possibleAlternateItems = inventoryItemService.getPossibleAlternateItems(result.inventoryItemInstance)
        def suppliers = Supplier.findAllByIsActive(true).sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) }
        def manufacturers = Manufacturer.findAllByIsActive(true).sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) }
        render(view:'edit', model:[inventoryItemInstance: result.inventoryItemInstance.attach(),
                                                possibleAlternateItems: possibleAlternateItems,
                                                suppliers: suppliers,
                                                manufacturers: manufacturers])
    }

    def create = {
        def result = inventoryItemService.create(params)
        def suppliers = Supplier.findAllByIsActive(true).sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) }
        def manufacturers = Manufacturer.findAllByIsActive(true).sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) }

        if(!result.error)
            return [inventoryItemInstance: result.inventoryItemInstance,
                            suppliers: suppliers,
                            manufacturers: manufacturers]

        //flash.errorMessage = g.message(code: result.error.code, args: result.error.args)
        redirect(action: search)
    }

    def save = {
        def result = inventoryItemService.save(params)

        if(!result.error) {
            flash.message = g.message(code: "default.create.success", args: ["InventoryItem", result.inventoryItemInstance.id])
            redirect(action:show, id: result.inventoryItemInstance.id)
            return
        }

        def suppliers = Supplier.findAllByIsActive(true).sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) }
        def manufacturers = Manufacturer.findAllByIsActive(true).sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) }

        //flash.errorMessage = g.message(code: result.error.code, args: result.error.args)
        render(view:'create', model:[inventoryItemInstance: result.inventoryItemInstance,
                                                    suppliers: suppliers,
                                                    manufacturers: manufacturers])
    }

    /**
    * Handles the use inventory item form submit in the show view.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def useInventoryItem = {

        params.inventoryMovementType = InventoryMovementType.get(1) // Set type to "Used".
        def result = inventoryMovementService.move(params)

        if(!result.error) {
            flash.message = "Inventory Movement for ${result.inventoryMovementInstance.inventoryItem.name.encodeAsHTML()} created."
            session.inventoryMovementTaskId = null
            redirect(controller: "taskDetailed",
                            action: "show",
                            id: result.taskId,
                            params: [showTab: "showInventoryTab"])
            // Success.
            return
        }

        // Prepare data for the show view.
        def p = [:]
        p.id = result.inventoryMovementInstance.inventoryItem?.id
        def r = inventoryItemService.show(p)

        // Render show view if data was successfully prepared.
        if(!r.error) {
            def model = [ inventoryItemInstance: r.inventoryItemInstance,
                                    inventoryMovementList: r.inventoryMovementList,
                                    inventoryMovementListTotal: r.inventoryMovementListTotal,
                                    inventoryMovementListMax: r.inventoryMovementListMax,
                                    inventoryItemPurchases: r.inventoryItemPurchases,
                                    inventoryItemPurchasesTotal: r.inventoryItemPurchasesTotal,
                                    showTab: r.showTab]

            model.inventoryMovementInstance = result.inventoryMovementInstance // This will pass in the errors.

            render(view: 'show', model: model)
            return
        }

        // Could not prepare data for show view so doing the next best thing.
        flash.errorMessage = g.message(code: r.error.code, args: r.error.args)
        redirect(action:search)

    } // useInventoryItem

    /**
    * Clear the use inventory item form in the show view.
    * Accomplished by clearing the session variable and ajax.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def clearUseInventoryItem = {
            session.inventoryMovementTaskId = null
            render ''
    }

} // end of class
